/* ===== Luxury Carousel – loop + drag + snap (conflict-proof) ===== */
(function(){
    const root = document.getElementById('luxuryCarousel');
    if (!root) return;
  
    const viewport = root.querySelector('.viewport');
    const track    = root.querySelector('.track');
    const originals= Array.from(track.querySelectorAll('.lux-slide'));
    const prevBtn  = root.querySelector('.arrow.prev');
    const nextBtn  = root.querySelector('.arrow.next');
    const dotsWrap = root.querySelector('.dots');
  
    const N = originals.length;
    if (!N) return;
  
    // Clone left & right so first/last also show peeks (visual loop)
    const leftFrag  = document.createDocumentFragment();
    const rightFrag = document.createDocumentFragment();
    originals.forEach(s => { const c = s.cloneNode(true); c.dataset.clone='1'; rightFrag.appendChild(c); });
    originals.forEach(s => { const c = s.cloneNode(true); c.dataset.clone='1'; leftFrag.appendChild(c);  });
    track.prepend(leftFrag);
    track.append(rightFrag);
  
    let slides   = Array.from(track.querySelectorAll('.lux-slide'));
    const BASE   = N;                               // first original index in slides[]
    let absIndex = BASE + Math.floor(N/2);          // absolute index (incl. clones)
    let realIndex= (absIndex - BASE + N) % N;       // 0..N-1
  
    // Dots (originals only)
    dotsWrap.innerHTML = '';
    const dots = originals.map((_, i) => {
      const d = document.createElement('button');
      d.className = 'dot'; d.type = 'button';
      d.addEventListener('click', () => goToReal(i));
      dotsWrap.appendChild(d);
      return d;
    });
  
    // Helpers
    const curTX = () => {
      const m = getComputedStyle(track).transform;
      return m !== 'none' ? new DOMMatrix(m).m41 : 0;
    };
    const setTX = (x, instant=false) => {
      track.style.transitionDuration = instant ? '0ms' : '';
      track.style.transform = `translateX(${x}px)`;
    };
  
    function setActiveByAbs(a){
      slides.forEach(s=>s.classList.remove('is-active'));
      slides[a].classList.add('is-active');
      realIndex = ((a - BASE) % N + N) % N;
      dots.forEach((d,i)=>d.classList.toggle('is-active', i===realIndex));
    }
  
    // Center on an element relative to viewport center
    function centerOn(el, {instant=false}={}){
      const vRect = viewport.getBoundingClientRect();
      const eRect = el.getBoundingClientRect();
      const delta = (eRect.left + eRect.width/2) - (vRect.left + vRect.width/2);
      setTX(curTX() - delta, instant);
    }
  
    function goToAbs(a, {instant=false}={}){
      absIndex = a;
      centerOn(slides[absIndex], {instant});
      setActiveByAbs(absIndex);
    }
    function goToReal(r, opts={}){
      const targetAbs = BASE + ((r % N + N) % N);
      goToAbs(targetAbs, opts);
    }
  
    // If we land on clone band, jump instantly to its original
    function normalizeIfClone(){
      if (absIndex < BASE || absIndex >= BASE + N){
        absIndex = BASE + realIndex;
        centerOn(slides[absIndex], {instant:true});
      }
    }
  
    // Arrows
    prevBtn && prevBtn.addEventListener('click', ()=> goToReal(realIndex - 1));
    nextBtn && nextBtn.addEventListener('click', ()=> goToReal(realIndex + 1));
  
    // Drag / swipe + snap
    let isDrag=false, startX=0, startTX=0;
    track.addEventListener('pointerdown', e=>{
      isDrag = true; startX = e.clientX; startTX = curTX();
      track.style.transitionDuration = '0ms';
      track.setPointerCapture(e.pointerId);
    });
    addEventListener('pointermove', e=>{
      if(!isDrag) return;
      setTX(startTX + (e.clientX - startX), true);
    });
    addEventListener('pointerup', ()=>{
      if(!isDrag) return; isDrag=false;
  
      const vRect = viewport.getBoundingClientRect();
      const vCenter = vRect.left + vRect.width/2;
  
      let best = {a:0, d:Infinity};
      slides.forEach((el, a)=>{
        const c = el.getBoundingClientRect().left + el.offsetWidth/2;
        const d = Math.abs(c - vCenter);
        if (d < best.d) best = {a, d};
      });
      goToAbs(best.a);
    });
  
    track.addEventListener('transitionend', normalizeIfClone);
    new ResizeObserver(()=> goToAbs(absIndex, {instant:true})).observe(viewport);
  
    // Start centered on a middle original
    window.addEventListener('load', ()=>{ goToAbs(absIndex, {instant:true}); });
  })();
  