// Navigation functionality
const navToggle = document.querySelector('.nav-toggle');
const headerLeft = document.querySelector('.header-left');
const headerRight = document.querySelector('.header-right');
const navClose = document.querySelector('.nav-close');
const fullscreenNav = document.querySelector('.fullscreen-nav');
const body = document.body;
const header = document.querySelector('.header');

// Initialize AOS (Animate On Scroll)
document.addEventListener('DOMContentLoaded', function() {
  AOS.init({
    easing: 'ease-in-out'
  });
});

// Handle menu items with children
document.querySelectorAll('.nav-parent.has-children > a.parent').forEach(parentLink => {
  parentLink.addEventListener('click', (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    // Get the child identifier from the clicked parent
    const childId = parentLink.getAttribute('data-child');
    
    // If the clicked parent is already active, deactivate it
    if (parentLink.classList.contains('active')) {
      parentLink.classList.remove('active');
      const targetChildren = document.querySelector(`.nav-children[data-parent="${childId}"]`);
      if (targetChildren) {
        targetChildren.classList.remove('active');
      }
      return;
    }
    
    // Hide all nav-children first
    document.querySelectorAll('.nav-children').forEach(navChild => {
      navChild.classList.remove('active');
    });
    
    // Remove active class from all parent links
    document.querySelectorAll('.nav-parent.has-children > a.parent').forEach(link => {
      link.classList.remove('active');
    });
    
    // Show the corresponding nav-children and activate the parent link
    const targetChildren = document.querySelector(`.nav-children[data-parent="${childId}"]`);
    if (targetChildren) {
      targetChildren.classList.add('active');
      parentLink.classList.add('active');
    }
  });
});

// Close child menus when clicking outside
document.addEventListener('click', (e) => {
  // Don't close if clicking inside the fullscreen nav or on the header
  if (e.target.closest('.fullscreen-nav') || e.target.closest('.header')) {
    return;
  }
  
  if (!e.target.closest('.nav-parent.has-children')) {
    document.querySelectorAll('.nav-children').forEach(navChild => {
      navChild.classList.remove('active');
    });
    document.querySelectorAll('.nav-parent.has-children > a.parent').forEach(link => {
      link.classList.remove('active');
    });
  }
});

function openServicesIfMobile() {
  const isMobile = window.innerWidth <= 767;

  if (isMobile) {
    const servicesLink = document.querySelector('.nav-parent.has-children > a.parent[data-child="services"]');

    if (servicesLink && !servicesLink.classList.contains('active')) {
      servicesLink.click();
    }
  }
}

function toggleNav() {
  const isOpening = !fullscreenNav.classList.contains('active');
  
  if (isOpening) {
    // Opening menu
    fullscreenNav.classList.add('active');
    headerLeft.classList.add('active');
    navClose.classList.add('active');
    body.classList.add('menu-active');

    setTimeout(openServicesIfMobile, 10);
  } else {
    // Closing menu
    fullscreenNav.classList.remove('active');
    headerLeft.classList.remove('active');
    navClose.classList.remove('active');
    body.classList.remove('menu-active');
  }
}

if (navToggle) {
  navToggle.addEventListener('click', toggleNav);
}

if (navClose) {
  navClose.addEventListener('click', toggleNav);
}

// Close nav when clicking a link
document.querySelectorAll('.fullscreen-nav a:not(.parent)').forEach(link => {
  link.addEventListener('click', () => {
    // Use toggleNav() to properly handle all state changes including button states
    if (fullscreenNav.classList.contains('active')) {
      toggleNav();
    }
  });
});

// Close nav when pressing Escape key
document.addEventListener('keydown', (e) => {
  if (e.key === 'Escape' && fullscreenNav.classList.contains('active')) {
    toggleNav();
  }
});

// Smooth scrolling for navigation links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
  anchor.addEventListener('click', function (e) {
    e.preventDefault();
    const target = document.querySelector(this.getAttribute('href'));
    if (target) {
      const headerHeight = document.querySelector('.header').offsetHeight;
      const targetPosition = target.offsetTop - headerHeight;
      
      window.scrollTo({
        top: targetPosition,
        behavior: 'smooth'
      });
    }
  });
});

// Header scroll effect
let lastScrollTop = 0;
const scrollThreshold = 50;

// Function to check scroll position and apply scrolled class
function checkScrollPosition() {
  const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
  
  if (scrollTop > scrollThreshold) {
    body.classList.add('scrolled');
  } else {
    body.classList.remove('scrolled');
  }
}

window.addEventListener('scroll', () => {
  const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
  
  // Add scrolled class when page is scrolled
  if (scrollTop > scrollThreshold) {
    body.classList.add('scrolled');
  } else {
    body.classList.remove('scrolled');
  }
  
  // Mobile scroll detection for header elements
  if (window.innerWidth <= 767) {
    const headerLeft = document.querySelector('.header-left');
    const headerRight = document.querySelector('.header-right');
    
    // Only apply scroll detection if elements exist and menu is not active
    if (headerLeft && headerRight && !body.classList.contains('menu-active')) {
      if (scrollTop > lastScrollTop && scrollTop > 100) {
        // Scrolling down - hide header elements
        headerLeft.classList.add('header-hidden');
        headerRight.classList.add('header-hidden');
        console.log('Hiding headers - scrolling down');
      } else if (scrollTop < lastScrollTop) {
        // Scrolling up - show header elements
        headerLeft.classList.remove('header-hidden');
        headerRight.classList.remove('header-hidden');
        console.log('Showing headers - scrolling up');
      }
    }
  }
  
  lastScrollTop = scrollTop;
});

// Check scroll position on page load and apply appropriate class
document.addEventListener('DOMContentLoaded', () => {
  checkScrollPosition();
});

// CTA Button interactions
document.querySelectorAll('.cta-button').forEach(button => {
  button.addEventListener('click', function(e) {
    // Add ripple effect
    const ripple = document.createElement('span');
    const rect = this.getBoundingClientRect();
    const size = Math.max(rect.width, rect.height);
    const x = e.clientX - rect.left - size / 2;
    const y = e.clientY - rect.top - size / 2;
    
    ripple.style.width = ripple.style.height = size + 'px';
    ripple.style.left = x + 'px';
    ripple.style.top = y + 'px';
    ripple.classList.add('ripple');
    
    this.appendChild(ripple);
    
    setTimeout(() => {
      ripple.remove();
    }, 600);
    
    // Handle button actions (placeholder)
    console.log('CTA Button clicked:', this.textContent);
  });
});

// Add ripple effect styles
const style = document.createElement('style');
style.textContent = `
  .cta-button {
    position: relative;
    overflow: hidden;
  }
  
  .ripple {
    position: absolute;
    border-radius: 50%;
    background: rgba(255, 255, 255, 0.3);
    transform: scale(0);
    animation: ripple 0.6s ease-out;
    pointer-events: none;
  }
  
  @keyframes ripple {
    to {
      transform: scale(2);
      opacity: 0;
    }
  }
`;
document.head.appendChild(style);


// Banner Slideshow
function initBannerSlideshow() {
  const slides = document.querySelectorAll('.banner#hero .banner-slide');
  if (slides.length === 0) return; // Exit if no slides found
  
  let currentSlide = 0;
  const slideInterval = 4000; // Change slide every 4 seconds for image slides
  let slideTimer = null;

  function nextSlide() {
    // Pause video in current slide if it exists
    const currentVideo = slides[currentSlide].querySelector('video');
    if (currentVideo) {
      currentVideo.pause();
    }
    
    // Remove active class from current slide
    slides[currentSlide].classList.remove('active');
    
    // Update current slide index
    currentSlide = (currentSlide + 1) % slides.length;
    
    // Add active class to new current slide
    slides[currentSlide].classList.add('active');
    
    // Play video in new slide if it exists
    const newVideo = slides[currentSlide].querySelector('video');
    if (newVideo) {
      newVideo.currentTime = 0; // Reset to beginning
      newVideo.play().catch(e => console.log('Video autoplay prevented:', e));
      
      // Wait for video to finish before moving to next slide immediately
      newVideo.addEventListener('ended', function onVideoEnd() {
        newVideo.removeEventListener('ended', onVideoEnd); // Remove listener to avoid duplicates
        nextSlide(); // Immediately go to next slide
      }, { once: true });
    } else {
      // If it's an image slide, set up timer for next slide
      setupNextSlideTimer();
    }
  }

  function setupNextSlideTimer() {
    // Clear any existing timer
    if (slideTimer) {
      clearTimeout(slideTimer);
    }
    
    // Set timer for next slide
    slideTimer = setTimeout(nextSlide, slideInterval);
  }

  // Set initial state
  slides[0].classList.add('active');
  
  // Play video in first slide if it's a video slide
  const firstVideo = slides[0].querySelector('video');
  if (firstVideo) {
    firstVideo.play().catch(e => console.log('Video autoplay prevented:', e));
    
    // Wait for video to finish before moving to next slide immediately
    firstVideo.addEventListener('ended', function onFirstVideoEnd() {
      firstVideo.removeEventListener('ended', onFirstVideoEnd); // Remove listener to avoid duplicates
      nextSlide(); // Immediately go to next slide
    }, { once: true });
  } else {
    // If first slide is an image, set up timer for next slide
    setupNextSlideTimer();
  }

  // Only start the slideshow if there's more than one slide
  if (slides.length > 1) {
    // Initial timer setup is handled above based on first slide type
  }
}

// Scroll Down Button Functionality
document.querySelector('.scroll-down-btn').addEventListener('click', () => {
  const textSection = document.querySelector('.text-section-1');
  const duration = 1000; // Duration in milliseconds
  const start = window.pageYOffset;
  const scrollDistance = textSection.offsetTop - start;
  const startTime = performance.now();

  function scrollStep(timestamp) {
    const currentTime = timestamp - startTime;
    const progress = Math.min(currentTime / duration, 1);
    
    // Easing function for smooth acceleration and deceleration
    const easeInOutCubic = progress => {
      return progress < 0.5
        ? 4 * progress * progress * progress
        : 1 - Math.pow(-2 * progress + 2, 3) / 2;
    };

    window.scrollTo(0, start + (scrollDistance * easeInOutCubic(progress)));

    if (progress < 1) {
      requestAnimationFrame(scrollStep);
    }
  }

  requestAnimationFrame(scrollStep);
});

// Initialize banner slideshow when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  initBannerSlideshow();
  initVideoModal();
  wrapLettersInSpans();
});

// Initialize Splide Carousel for Services
document.addEventListener('DOMContentLoaded', () => {
  const servicesSplide = new Splide('.services-splide', {
    type: 'loop',
    perPage: 3,
    perMove: 1,
    focus: 'center',
    gap: '1rem',
    padding: { left: '10%', right: '10%' },
    breakpoints: {
      1200: {
        perPage: 2,
        padding: { left: '5%', right: '5%' }
      },
      768: {
        perPage: 1,
        padding: { left: '5%', right: '5%' }
      }
    },
    autoplay: true,
    interval: 4000,
    pauseOnHover: true,
    pauseOnFocus: true,
    arrows: true,
    pagination: false,
    classes: {
      arrows: 'splide__arrows',
      arrow: 'splide__arrow',
      prev: 'splide__arrow--prev',
      next: 'splide__arrow--next'
    }
  });

  servicesSplide.mount();

  // Initialize Splide Carousel for Promotions
  const promotionsSplide = new Splide('.promotions-splide', {
    type: 'loop',
    perPage: 3,
    gap: '2rem',
    pagination: true,
    arrows: true,
    autoplay: true,
    interval: 4000,
    pauseOnHover: true,
    pauseOnFocus: true,
    breakpoints: {
      768: {
        perPage: 1,
      }
    },
    classes: {
      arrows: 'splide__arrows',
      arrow: 'splide__arrow',
      prev: 'splide__arrow--prev',
      next: 'splide__arrow--next'
    }
  });

  promotionsSplide.mount();
});


// Stacked Card Carousel
const stack = document.querySelector('.stack');
const nextBtn = document.querySelector('.nav-btn.next');
const prevBtn = document.querySelector('.nav-btn.prev');

const cards = Array.from(stack.children).reverse().filter((child) => child.classList.contains('card'));

cards.forEach((card) => stack.appendChild(card));

function moveCard(direction) {
  if (direction === 'next') {
    const lastCard = stack.lastElementChild;
    if (lastCard.classList.contains("card")) {
      lastCard.classList.add("swap-next");
      setTimeout(() => {
        lastCard.classList.remove("swap-next");
        stack.insertBefore(lastCard, stack.firstElementChild);
      }, 1200);
    }
  } else if (direction === 'prev') {
    const firstCard = stack.firstElementChild;
    if (firstCard.classList.contains("card")) {
      firstCard.classList.add("swap-prev");
      setTimeout(() => {
        firstCard.classList.remove("swap-prev");
        stack.appendChild(firstCard);
      }, 1200);
    }
  }
}

let autoplayInterval = setInterval(() => moveCard('next'), 4000);

// Add click event listeners to both buttons
nextBtn.addEventListener("click", () => moveCard('next'));
prevBtn.addEventListener("click", () => moveCard('prev'));

// Mobile Events Carousel
function initMobileEventsCarousel() {
  const mobileCarousel = document.querySelector('.mobile-events-carousel');
  if (!mobileCarousel) return;

  const track = mobileCarousel.querySelector('.carousel-track');
  const slides = mobileCarousel.querySelectorAll('.carousel-slide');
  const prevBtn = mobileCarousel.querySelector('.nav-btn.prev');
  const nextBtn = mobileCarousel.querySelector('.nav-btn.next');
  const dots = mobileCarousel.querySelectorAll('.carousel-dot');

  let currentSlide = 0;
  const totalSlides = slides.length;

  function updateCarousel() {
    const slideWidth = 100; // 100% width
    track.style.transform = `translateX(-${currentSlide * slideWidth}%)`;
    
    // Update dots
    dots.forEach((dot, index) => {
      dot.classList.toggle('active', index === currentSlide);
    });
  }

  function nextSlide() {
    currentSlide = (currentSlide + 1) % totalSlides;
    updateCarousel();
  }

  function prevSlide() {
    currentSlide = (currentSlide - 1 + totalSlides) % totalSlides;
    updateCarousel();
  }

  function goToSlide(index) {
    currentSlide = index;
    updateCarousel();
  }

  // Event listeners
  if (nextBtn) {
    nextBtn.addEventListener('click', nextSlide);
  }

  if (prevBtn) {
    prevBtn.addEventListener('click', prevSlide);
  }

  // Dot navigation
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => goToSlide(index));
  });

  // Touch/swipe support
  let startX = 0;
  let endX = 0;

  track.addEventListener('touchstart', (e) => {
    startX = e.touches[0].clientX;
  });

  track.addEventListener('touchend', (e) => {
    endX = e.changedTouches[0].clientX;
    const diff = startX - endX;
    
    if (Math.abs(diff) > 50) { // Minimum swipe distance
      if (diff > 0) {
        nextSlide(); // Swipe left
      } else {
        prevSlide(); // Swipe right
      }
    }
  });

  // Autoplay
  let mobileAutoplayInterval = setInterval(nextSlide, 4000);

  // Pause autoplay on hover/touch
  mobileCarousel.addEventListener('mouseenter', () => {
    clearInterval(mobileAutoplayInterval);
  });

  mobileCarousel.addEventListener('mouseleave', () => {
    mobileAutoplayInterval = setInterval(nextSlide, 4000);
  });

  // Initialize
  updateCarousel();
}

// Initialize mobile carousel when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  initMobileEventsCarousel();
});

// Mobile Footer Accordion
function initMobileFooterAccordion() {
  const accordionHeaders = document.querySelectorAll('.footer-section-header');
  
  accordionHeaders.forEach(header => {
    header.addEventListener('click', () => {
      const content = header.nextElementSibling;
      const icon = header.querySelector('.accordion-icon');
      
      // Close all other accordion sections
      accordionHeaders.forEach(otherHeader => {
        if (otherHeader !== header) {
          otherHeader.classList.remove('active');
          const otherContent = otherHeader.nextElementSibling;
          otherContent.classList.remove('active');
          // Reset maxHeight for smooth closing animation
          otherContent.style.maxHeight = '0px';
        }
      });
      
      // Toggle current section
      header.classList.toggle('active');
      content.classList.toggle('active');
      
      // Smooth animation for content height
      if (content.classList.contains('active')) {
        const contentHeight = content.scrollHeight;
        content.style.maxHeight = contentHeight + 'px';
      } else {
        content.style.maxHeight = '0px';
      }
    });
  });
}

// Initialize mobile footer accordion when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  initMobileFooterAccordion();
});

// Wrap letters in Tradewinds section titles with spans
function wrapLettersInSpans() {
  console.log('wrapLettersInSpans function called');
  
  // Try different selectors to find the titles
  const tradewindsTitles = document.querySelectorAll('.text-section .section-title');
  console.log('Found titles with .tradewinds .text-section .section-title:', tradewindsTitles.length);
  
  if (tradewindsTitles.length === 0) {
    // Try alternative selectors
    const allTitles = document.querySelectorAll('.section-title');
    console.log('Found all .section-title elements:', allTitles.length);
    
    allTitles.forEach(title => {
      console.log('Title text:', title.textContent);
      console.log('Title parent classes:', title.parentElement.className);
    });
  }
  
  tradewindsTitles.forEach((title, index) => {
    console.log(`Processing title ${index}:`, title.textContent);
    
    if (!title.classList.contains('letters-wrapped')) {
      const html = title.innerHTML;
      console.log('Original HTML:', html);
      
      // Split by HTML tags and text content
      const parts = html.split(/(<[^>]*>)/);
      const wrappedParts = parts.map(part => {
        // If it's an HTML tag, keep it as is
        if (part.startsWith('<') && part.endsWith('>')) {
          return part;
        }
        // If it's text content, wrap each letter
        else if (part.trim() !== '') {
          let letterIndex = 0;
          return part.split('').map(letter => {
            if (letter === ' ') {
              return '<span class="letter-space">&nbsp;</span>';
            } else {
              const span = `<span class="letter" style="--letter-index: ${letterIndex}">${letter}</span>`;
              letterIndex++;
              return span;
            }
          }).join('');
        }
        return part;
      });
      
      const wrappedHTML = wrappedParts.join('');
      console.log('Wrapped HTML:', wrappedHTML);
      title.innerHTML = wrappedHTML;
      title.classList.add('letters-wrapped');
      console.log('Letters wrapped successfully');
    } else {
      console.log('Title already wrapped');
    }
  });
}

// Initialize letter wrapping when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  wrapLettersInSpans();
});

// Video Modal Functionality
function initVideoModal() {
  const videoModal = document.getElementById('videoModal');
  if (!videoModal) return; // Exit if modal doesn't exist
  
  const videoModalClose = document.querySelector('.video-modal-close');
  const modalVideo = videoModal.querySelector('video');
  
  // Get video elements from both banner slider and separate video banner
  const bannerSliderVideo = document.querySelector('.banner-slide.video-slide video');
  const bannerVideo = document.querySelector('.video-banner video');
  const videoPlayBtn = document.querySelector('.video-slide .video-play-btn');
  const bannerVideoPlayBtn = document.querySelector('.video-banner .video-play-btn');

  // Debug video loading for banner slider video
  if (bannerSliderVideo) {
    bannerSliderVideo.addEventListener('loadstart', () => console.log('Banner slider video loading started'));
    bannerSliderVideo.addEventListener('canplay', () => console.log('Banner slider video can play'));
    bannerSliderVideo.addEventListener('error', (e) => console.error('Banner slider video error:', e));
    bannerSliderVideo.addEventListener('loadeddata', () => console.log('Banner slider video data loaded'));
    
    // Check if video is actually playing
    setInterval(() => {
      if (bannerSliderVideo.readyState >= 2) {
        console.log('Banner slider video ready state:', bannerSliderVideo.readyState);
      }
    }, 2000);
  } else {
    console.log('Banner slider video element not found');
  }

  // Debug video loading for separate video banner
  if (bannerVideo) {
    bannerVideo.addEventListener('loadstart', () => console.log('Video banner loading started'));
    bannerVideo.addEventListener('canplay', () => console.log('Video banner can play'));
    bannerVideo.addEventListener('error', (e) => console.error('Video banner error:', e));
    bannerVideo.addEventListener('loadeddata', () => console.log('Video banner data loaded'));
    
    // Check if video is actually playing
    setInterval(() => {
      if (bannerVideo.readyState >= 2) {
        console.log('Video banner ready state:', bannerVideo.readyState);
      }
    }, 2000);
  } else {
    console.log('Video banner element not found');
  }

  // Open modal when play button is clicked (banner slider video)
  if (videoPlayBtn) {
    videoPlayBtn.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      openVideoModal();
    });
  }

  // Open modal when video overlay is clicked (banner slider video)
  const videoOverlay = document.querySelector('.video-slide .video-overlay');
  if (videoOverlay) {
    videoOverlay.addEventListener('click', function(e) {
      if (e.target !== videoPlayBtn) {
        openVideoModal();
      }
    });
  }

  // Open modal when play button is clicked (separate video banner)
  if (bannerVideoPlayBtn) {
    bannerVideoPlayBtn.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      openVideoModal();
    });
  }

  // Open modal when video overlay is clicked (separate video banner)
  const bannerVideoOverlay = document.querySelector('.video-banner .video-overlay');
  if (bannerVideoOverlay) {
    bannerVideoOverlay.addEventListener('click', function(e) {
      if (e.target !== bannerVideoPlayBtn) {
        openVideoModal();
      }
    });
  }

  // Close modal when close button is clicked
  if (videoModalClose) {
    videoModalClose.addEventListener('click', function() {
      closeVideoModal();
    });
  }

  // Close modal when clicking outside the video
  videoModal.addEventListener('click', function(e) {
    if (e.target === videoModal) {
      closeVideoModal();
    }
  });

  // Close modal with Escape key
  document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && videoModal.classList.contains('active')) {
      closeVideoModal();
    }
  });

  function openVideoModal() {
    videoModal.classList.add('active');
    document.body.style.overflow = 'hidden'; // Prevent background scrolling
    
    // Pause all videos and play modal video
    if (bannerSliderVideo) {
      bannerSliderVideo.pause();
    }
    if (bannerVideo) {
      bannerVideo.pause();
    }
    if (modalVideo) {
      modalVideo.currentTime = 0; // Start from beginning
      modalVideo.play();
    }
  }

  function closeVideoModal() {
    videoModal.classList.remove('active');
    document.body.style.overflow = ''; // Restore scrolling
    
    // Pause modal video and resume appropriate video
    if (modalVideo) {
      modalVideo.pause();
    }
    
    // Resume the video that was playing before modal opened
    // Check which video slide is currently active
    const activeSlide = document.querySelector('.banner-slide.active');
    if (activeSlide && activeSlide.classList.contains('video-slide') && bannerSliderVideo) {
      bannerSliderVideo.play();
    } else if (bannerVideo) {
      bannerVideo.play();
    }
  }
}


